# -*- coding: utf-8 -*-
"""
    Action d'un filtre d'ordre 2 sur un signal carré
"""

#### Importation des bibliothèques utiles

import numpy as np
import matplotlib.pyplot as plt

############       DIAGRAMME DE BODE DU FILTRE D'ORDRE 2       ###################

# Constantes du filtre d'ordre 2
#---------------------------------------------------------------
f0 = 1000        #  f0 (Hz) = fréquence centrale du filtre
Q = 100          #  Q : facteur de qualité
H0 = 1           #  gain
#---------------------------------------------------------------


k = 2000                                # nombre de points
f = np.logspace(-2, 2, k+1)*f0          # k intervalles, entre 10^-2f0 et 10^2f0 ; k+1 valeurs


#Définition de la fonction de transfert selon la nature du filtre choisir num

def H(H0, f0, Q, f):                  # ordre des variables arbitraire, à conserver à chaque appel de fonction
    # num = 1                         # passe-bas 
    num = 1j*f/(Q*f0)                 # passe-bande
    # num = (1j*f/f0)**2              # passe-haut   
    return H0*num/(1 + 1j*f/(Q*f0) + (1j*f/f0)**2)


#Définition du diagramme de Bode
    
GdB = 20*np.log10(np.abs(H(H0, f0, Q, f)))
phi = np.angle(H(H0, f0, Q, f), deg=True)


#Tracé du diagramme de Bode
#GdB et phi en concordance de fréquence
plt.figure(1)
plt.subplot(211)
plt.semilogx(f, GdB)
plt.title('Diagramme de Bode, H0=' + str(H0) + ', Q =' + str(Q) + 'f0 =' + str(f0) + 'Hz')
plt.ylabel('GdB')
plt.grid()

plt.subplot(212)
plt.semilogx(f, phi)
plt.ylabel('phase (°)')
plt.xlabel('f (Hz)')
plt.grid()

#plt.show()


############       SPECTRES D'ENTREE ET SORTIE       ###################

"""
On fabrique un signal carré à partir de sa décomposition spectrale sous forme d'une série de sinus
Ue = Uo + U1.sin(2pift) + U1/3.sin(2*pi*3f*t) + ...
si n = 1 on retrouve un signal sinusoidal
"""

fe = 1000                                    # fréquence fondamentale du signal d'entrée (en Hz)     
n = 5		                                 # harmoniques impairs uniquement
B = [2*i + 1 for i in range(n)]              # B est la liste des n premiers entiers impairs
F = np.array([0] + B)*fe                     # F est le tableau des n+1 fréquences non nulles du spectre
U0 = 2                                       # valeur moyenne du signal (V)
U1 = 1                                       # amplitude du fondamental (V)
Ue = np.array([U0] + list(U1/np.array(B)))   # amplitudes des composantes (V)
phie = np.array([0] + [-90]*n)	             # phases à l'origine (°)


# Amplitude et phase d'une harmonique
def ordre_2(H0, f0, Q, F, U) :              # F est le tableau des fréquences
	Hs   = H(H0, f0, Q, F)                  # U est ici une variable muette
	Us   = np.abs(Hs) * U
	phis = np.angle(Hs, deg=True) + phie
	return Us, phis

## Superposition des spectres d'entrée et sortie pour f0 choisie
Us, phis = ordre_2(H0, f0, Q, F, Ue)	    # listes des amplitudes de sortie et déphasages
                                            # On affecte Ue à U
largeur = fe/12                             # largeur des raies

plt.figure(2)
ax1 = plt.subplot(211)					                            # figure 2 lignes, 1 colonne
ax1.bar(F - largeur/2, Ue, largeur, label='Uem')                    # spectre amplitude entrée
ax1.bar(F + largeur/2, Us, largeur, label= 'Usm pour f0 =' +str(f0) + 'Hz') # spectre amplitude sortie 
ax1.set_ylabel('U (V)')
ax1.set_title('Spectres en amplitude et en phase')
ax1.set_xticks(F)
ax1.legend()
ax1.grid()

ax2 = plt.subplot(212)
ax2.bar(F - largeur/2, phie, largeur, label='phie')                     # spectre phase entrée
ax2.bar(F + largeur/2, phis, largeur, label='phis pour f0 =' + str(f0) + 'Hz')  # spectre phase sortie
ax2.set_xticks(F)
ax2.set_xlabel('f (Hz)')
ax2.set_ylabel('phase (°)')
ax2.legend(loc='lower center')
ax2.grid()



##############   Tracés de ue(t) et us(t) 

t = np.linspace(0, 4/fe, 1001)   # tracé entre t = 0 et t(max) = 4T = 4/fe

def ue(t, F) :                   # calcule ue pour chaque instant t
    valeur_ue = 0
    for i in range(len(F)) :     # somme des composantes d'entrée à t
        valeur_ue = valeur_ue \
                    + Ue[i]*np.cos(2*np.pi*F[i]*t + phie[i]/180*np.pi) #rad
    return valeur_ue

def us(t, f0, Q, Us, phis) :     # calcule us pour chaque instant t (Us, phi calculés avant)
    valeur_us = 0
    for i in range(len(F)) :     # somme des composantes de sortie à t
        valeur_us = valeur_us \
                    + Us[i]*np.cos(2*np.pi*F[i]*t + phis[i]/180*np.pi)  #rad
    return valeur_us

plt.figure(3)
plt.plot(t, ue(t, F), label = 'ue')                  # t est une liste, ue(t) aussi
plt.plot(t, us(t, f0, Q, Us, phis), label = 'us')
plt.xlabel('t (s)'), 
plt.ylabel('amplitude (V)')
plt.legend()
plt.title('Tracés de ue(t) et us(t) pour fe =' + str(fe) + 'Hz et f0 =' + str(f0) + 'Hz')
plt.grid()
plt.show()
